// ================================
// Membership Application Form Script
// ================================

document.addEventListener('DOMContentLoaded', function() {
  const form = document.getElementById('membershipApplicationForm');
  const steps = document.querySelectorAll('.form-step');
  const progressSteps = document.querySelectorAll('.progress-step');
  const progressBar = document.getElementById('progressBar');
  const prevBtn = document.getElementById('prevBtn');
  const nextBtn = document.getElementById('nextBtn');
  const submitBtn = document.getElementById('submitBtn');
  
  let currentStep = 1;
  const totalSteps = steps.length;

  // Initialize
  updateProgressBar();
  setupEventListeners();

  function updateProgressBar() {
    const percentage = (currentStep / totalSteps) * 100;
    progressBar.style.width = percentage + '%';

    // Update progress steps
    progressSteps.forEach((step, index) => {
      const stepNum = index + 1;
      step.classList.remove('active', 'completed');
      
      if (stepNum === currentStep) {
        step.classList.add('active');
      } else if (stepNum < currentStep) {
        step.classList.add('completed');
      }
    });

    // Update form steps visibility
    steps.forEach((step, index) => {
      const stepNum = index + 1;
      if (stepNum === currentStep) {
        step.classList.add('active');
      } else {
        step.classList.remove('active');
      }
    });

    // Update button visibility
    if (currentStep === 1) {
      prevBtn.style.display = 'none';
      nextBtn.style.display = 'inline-flex';
      submitBtn.style.display = 'none';
    } else if (currentStep === totalSteps) {
      prevBtn.style.display = 'inline-flex';
      nextBtn.style.display = 'none';
      submitBtn.style.display = 'inline-flex';
    } else {
      prevBtn.style.display = 'inline-flex';
      nextBtn.style.display = 'inline-flex';
      submitBtn.style.display = 'none';
    }
  }

  function setupEventListeners() {
    // Next button
    nextBtn.addEventListener('click', function(e) {
      e.preventDefault();
      if (validateCurrentStep()) {
        currentStep++;
        if (currentStep > totalSteps) {
          currentStep = totalSteps;
        }
        updateProgressBar();
        window.scrollTo({ top: 0, behavior: 'smooth' });
      }
    });

    // Previous button
    prevBtn.addEventListener('click', function(e) {
      e.preventDefault();
      currentStep--;
      if (currentStep < 1) {
        currentStep = 1;
      }
      updateProgressBar();
      window.scrollTo({ top: 0, behavior: 'smooth' });
    });

    // Form submission
    form.addEventListener('submit', function(e) {
      e.preventDefault();
      if (validateCurrentStep()) {
        submitForm();
      }
    });

    // Marital status change
    const maritalStatusRadios = document.querySelectorAll('input[name="marital_status"]');
    maritalStatusRadios.forEach(radio => {
      radio.addEventListener('change', function() {
        const spouseSection = document.getElementById('spouseSection');
        const spouseSection2 = document.getElementById('spouseSection2');
        
        if (this.value === 'married') {
          spouseSection.style.display = 'grid';
          spouseSection2.style.display = 'grid';
        } else {
          spouseSection.style.display = 'none';
          spouseSection2.style.display = 'none';
          // Clear spouse fields
          document.getElementById('spouseName').value = '';
          document.getElementById('spouseIdType').value = '';
          document.getElementById('spouseIdNumber').value = '';
          document.getElementById('spousePhone').value = '';
          document.getElementById('spouseEmail').value = '';
        }
      });
    });

    // Occupation type change
    const occupationRadios = document.querySelectorAll('input[name="occupation_type"]');
    occupationRadios.forEach(radio => {
      radio.addEventListener('change', function() {
        const occupationOtherSection = document.getElementById('occupationOtherSection');
        
        if (this.value === 'other') {
          occupationOtherSection.style.display = 'grid';
        } else {
          occupationOtherSection.style.display = 'none';
          document.getElementById('occupationOther').value = '';
        }
      });
    });

    // Referral source change
    const referralCheckboxes = document.querySelectorAll('input[name="referral_source"]');
    referralCheckboxes.forEach(checkbox => {
      checkbox.addEventListener('change', updateReferralOtherDisplay);
    });

    // Add nominated kin button
    const addNominatedKinBtn = document.getElementById('addNominatedKinBtn');
    addNominatedKinBtn.addEventListener('click', addNominatedKinRow);

    // Remove nominated kin buttons (delegated)
    document.addEventListener('click', function(e) {
      if (e.target.classList.contains('remove-nominated-kin')) {
        e.preventDefault();
        e.target.closest('.nominated-kin-row').remove();
        updateRemoveButtons();
      }
    });
  }

  function updateReferralOtherDisplay() {
    const referralOtherSection = document.getElementById('referralOtherSection');
    const otherChecked = document.querySelector('input[name="referral_source"][value="other"]:checked');
    
    if (otherChecked) {
      referralOtherSection.style.display = 'grid';
    } else {
      referralOtherSection.style.display = 'none';
      document.getElementById('referralOther').value = '';
    }
  }

  function addNominatedKinRow() {
    const container = document.getElementById('nominatedKinsContainer');
    const row = document.createElement('div');
    row.className = 'nominated-kin-row';
    row.innerHTML = `
      <div class="form-row">
        <div class="form-group">
          <label class="form-label">Full Name</label>
          <input type="text" class="nominated-kin-name form-control" placeholder="Enter name">
        </div>

        <div class="form-group">
          <label class="form-label">Date of Birth</label>
          <input type="date" class="nominated-kin-dob form-control">
        </div>

        <div class="form-group">
          <label class="form-label">ID Number</label>
          <input type="text" class="nominated-kin-id form-control" placeholder="Enter ID number">
        </div>
      </div>

      <div class="form-row">
        <div class="form-group">
          <label class="form-label">Relationship</label>
          <input type="text" class="nominated-kin-relationship form-control" placeholder="Enter relationship">
        </div>

        <div class="form-group">
          <label class="form-label">Percentage Allocated (%)</label>
          <input type="number" class="nominated-kin-percentage form-control" placeholder="0" min="0" max="100" step="0.01">
        </div>

        <div class="form-group" style="display: flex; align-items: flex-end;">
          <button type="button" class="btn btn-sm btn-danger remove-nominated-kin">Remove</button>
        </div>
      </div>
    `;
    container.appendChild(row);
    updateRemoveButtons();
  }

  function updateRemoveButtons() {
    const rows = document.querySelectorAll('.nominated-kin-row');
    rows.forEach((row, index) => {
      const removeBtn = row.querySelector('.remove-nominated-kin');
      if (rows.length > 1) {
        removeBtn.style.display = 'block';
      } else {
        removeBtn.style.display = 'none';
      }
    });
  }

  function validateCurrentStep() {
    const currentStepElement = steps[currentStep - 1];
    const requiredFields = currentStepElement.querySelectorAll('[required]');
    let isValid = true;

    requiredFields.forEach(field => {
      // Check if field is visible (not in a hidden section)
      if (!isVisible(field)) {
        return;
      }

      if (field.type === 'radio' || field.type === 'checkbox') {
        const name = field.name;
        const checked = currentStepElement.querySelector(`input[name="${name}"]:checked`);
        if (!checked) {
          isValid = false;
          field.classList.add('is-invalid');
        } else {
          field.classList.remove('is-invalid');
        }
      } else if (field.type === 'file') {
        // File inputs are optional
        field.classList.remove('is-invalid');
      } else {
        if (!field.value.trim()) {
          isValid = false;
          field.classList.add('is-invalid');
        } else {
          field.classList.remove('is-invalid');
        }
      }
    });

    // Validate percentage allocation
    if (currentStep === 6) {
      validatePercentageAllocation();
    }

    if (!isValid) {
      alert('Please fill in all required fields.');
    }

    return isValid;
  }

  function isVisible(element) {
    return element.offsetParent !== null && 
           getComputedStyle(element).display !== 'none' &&
           element.closest('[style*="display: none"]') === null;
  }

  function validatePercentageAllocation() {
    const percentages = Array.from(document.querySelectorAll('.nominated-kin-percentage')).map(input => {
      const value = parseFloat(input.value) || 0;
      return value;
    });

    const total = percentages.reduce((sum, val) => sum + val, 0);

    if (total > 0 && total !== 100) {
      alert('Percentage allocation must equal 100%. Current total: ' + total + '%');
      return false;
    }

    return true;
  }

  function submitForm() {
    // Validate percentage allocation before submission
    if (!validatePercentageAllocation()) {
      return;
    }

    // Collect all form data
    const formData = new FormData(form);

    // Collect nominated kins data
    const nominatedKins = [];
    document.querySelectorAll('.nominated-kin-row').forEach(row => {
      const kin = {
        name: row.querySelector('.nominated-kin-name').value,
        dob: row.querySelector('.nominated-kin-dob').value,
        id: row.querySelector('.nominated-kin-id').value,
        relationship: row.querySelector('.nominated-kin-relationship').value,
        percentage: row.querySelector('.nominated-kin-percentage').value
      };
      if (kin.name) { // Only add if name is provided
        nominatedKins.push(kin);
      }
    });

    // Add nominated kins as JSON to form data
    formData.append('nominated_kins', JSON.stringify(nominatedKins));

    // Collect referral sources
    const referralSources = [];
    document.querySelectorAll('input[name="referral_source"]:checked').forEach(checkbox => {
      referralSources.push(checkbox.value);
    });
    formData.append('referral_sources', JSON.stringify(referralSources));

    // Submit form
    submitToServer(formData);
  }

  function submitToServer(formData) {
    const submitBtn = document.getElementById('submitBtn');
    const originalText = submitBtn.innerHTML;
    
    // Disable button and show loading state
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';

    fetch('admin/config/save_membership_application.php', {
      method: 'POST',
      body: formData
    })
    .then(response => response.json())
    .then(data => {
      if (data.success) {
        // Show success message
        showSuccessModal(data.application_id);
        // Reset form
        form.reset();
        currentStep = 1;
        updateProgressBar();
      } else {
        alert('Error submitting form: ' + data.message);
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
      }
    })
    .catch(error => {
      console.error('Error:', error);
      alert('Error submitting form. Please try again.');
      submitBtn.disabled = false;
      submitBtn.innerHTML = originalText;
    });
  }

  function showSuccessModal(applicationId) {
    const modal = document.createElement('div');
    modal.className = 'success-modal';
    modal.innerHTML = `
      <div class="modal-content">
        <div class="modal-header">
          <h2>Application Submitted Successfully</h2>
        </div>
        <div class="modal-body">
          <p>Thank you for your membership application!</p>
          <p>Your Application ID: <strong>${applicationId}</strong></p>
          <p>Please keep this for your records. Our team will review your application and contact you shortly.</p>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-primary" onclick="window.location.href='index.php'">Go to Home</button>
        </div>
      </div>
    `;
    
    document.body.appendChild(modal);
    modal.style.display = 'flex';
  }
});

// Success Modal Styles (added via JavaScript for convenience)
const style = document.createElement('style');
style.textContent = `
  .success-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    align-items: center;
    justify-content: center;
  }

  .modal-content {
    background-color: white;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.2);
    max-width: 500px;
    width: 90%;
  }

  .modal-header {
    padding: 20px;
    border-bottom: 2px solid #007b40;
  }

  .modal-header h2 {
    color: #007b40;
    margin: 0;
    font-size: 22px;
  }

  .modal-body {
    padding: 20px;
  }

  .modal-body p {
    margin: 0 0 15px 0;
    color: #333;
    line-height: 1.6;
  }

  .modal-body strong {
    color: #007b40;
  }

  .modal-footer {
    padding: 20px;
    border-top: 1px solid #eee;
    text-align: right;
  }
`;
document.head.appendChild(style);
