$(document).ready(function() {
    // Load loan types from database
    loadLoanTypes();

    // Event listeners
    $('#calculateBtn').on('click', calculateLoan);
    $('#resetBtn').on('click', resetCalculator);
    
    // Real-time validation
    $('#loanAmount, #loanTerm').on('input change', function() {
        if ($('#loanType').val() && $('#loanAmount').val() && $('#loanTerm').val()) {
            // You could add real-time calculation here if desired
        }
    });

    /**
     * Load loan types from the server
     */
    function loadLoanTypes() {
        $.ajax({
            url: 'admin/config/get_loan_types.php',
            type: 'GET',
            dataType: 'json',
            success: function(response) {
                if (response.success && response.data) {
                    let html = '<option value="">-- Select Loan Type --</option>';
                    $.each(response.data, function(index, loan) {
                        html += '<option value="' + loan.id + '" data-interest="' + loan.interest_rate_monthly + 
                                '" data-processing="' + loan.processing_fee + '" data-insurance="' + loan.insurance_fee_yearly + '">' + 
                                loan.loan_name + '</option>';
                    });
                    $('#loanType').html(html);
                } else {
                    showError('Failed to load loan types');
                }
            },
            error: function() {
                showError('Error loading loan types. Please try again.');
            }
        });
    }

    /**
     * Calculate the loan breakdown
     */
    function calculateLoan() {
        // Validate inputs
        if (!$('#loanType').val()) {
            showError('Please select a loan type');
            return;
        }
        if (!$('#loanAmount').val() || parseInt($('#loanAmount').val()) < 1000) {
            showError('Please enter a valid loan amount (minimum MWK 1,000)');
            return;
        }
        if (!$('#loanTerm').val() || parseInt($('#loanTerm').val()) < 1) {
            showError('Please enter a valid loan period');
            return;
        }

        // Get selected loan details
        const selectedOption = $('#loanType').find('option:selected');
        const interestRate = parseFloat(selectedOption.data('interest'));
        const processingFee = parseFloat(selectedOption.data('processing'));
        const insuranceRate = parseFloat(selectedOption.data('insurance'));

        const requestedAmount = parseFloat($('#loanAmount').val());
        const loanTerm = parseInt($('#loanTerm').val());

        // Calculate processing fee (deducted from requested amount)
        const processingFeeAmount = (requestedAmount * processingFee) / 100;
        const actualDisbursement = requestedAmount - processingFeeAmount;

        // Calculate monthly figures
        const monthlyPrincipal = actualDisbursement / loanTerm;
        const monthlyInterest = (actualDisbursement * interestRate) / 100;
        const monthlyInsurance = (requestedAmount * insuranceRate) / 100 / 12; // Annual insurance divided by 12 months
        const monthlyPayment = monthlyPrincipal + monthlyInterest + monthlyInsurance;

        // Calculate totals
        const totalInterest = monthlyInterest * loanTerm;
        const totalInsurance = monthlyInsurance * loanTerm;
        const totalRepayment = actualDisbursement + totalInterest + totalInsurance;
        const totalCost = totalInterest + totalInsurance;

        // Display results
        displayResults({
            requestedAmount: requestedAmount,
            processingFeeAmount: processingFeeAmount,
            processingFee: processingFee,
            actualDisbursement: actualDisbursement,
            loanTerm: loanTerm,
            interestRate: interestRate,
            monthlyPrincipal: monthlyPrincipal,
            monthlyInterest: monthlyInterest,
            monthlyInsurance: monthlyInsurance,
            monthlyPayment: monthlyPayment,
            totalPrincipal: actualDisbursement,
            totalInterest: totalInterest,
            totalInsurance: totalInsurance,
            totalRepayment: totalRepayment,
            totalCost: totalCost,
            insuranceRate: insuranceRate
        });
    }

    /**
     * Display calculation results
     */
    function displayResults(data) {
        $('#requestedAmount').text(formatCurrency(data.requestedAmount));
        $('#processingFee').text('- ' + formatCurrency(data.processingFeeAmount));
        $('#processingRate').text('(' + data.processingFee.toFixed(1) + '%)');
        $('#actualDisbursement').text(formatCurrency(data.actualDisbursement));
        
        $('#totalMonths').text(data.loanTerm + ' Month' + (data.loanTerm > 1 ? 's' : ''));
        $('#monthlyRate').text(data.interestRate.toFixed(1) + '%');
        $('#monthlyRatePercent').text('(' + data.interestRate.toFixed(1) + '%)');
        $('#monthlyPrincipal').text(formatCurrency(data.monthlyPrincipal));
        $('#interestOnPrincipal').text(formatCurrency(data.monthlyInterest));
        $('#monthlyInsurance').text(formatCurrency(data.monthlyInsurance));
        $('#monthlyPayment').text(formatCurrency(data.monthlyPayment));
        
        $('#totalPrincipal').text(formatCurrency(data.totalPrincipal));
        $('#totalInterest').text(formatCurrency(data.totalInterest));
        $('#totalInsuranceCost').text(formatCurrency(data.totalInsurance));
        $('#totalRepayment').text(formatCurrency(data.totalRepayment));
        $('#totalCost').text(formatCurrency(data.totalCost));

        // Show breakdown section with smooth animation
        $('#breakdownSection').slideDown(500);
        
        // Scroll to breakdown
        $('html, body').animate({
            scrollTop: $('#breakdownSection').offset().top - 100
        }, 800);
    }

    /**
     * Reset calculator
     */
    function resetCalculator() {
        $('#loanCalculatorForm')[0].reset();
        $('#breakdownSection').slideUp(300);
        $('#loanType').val('').trigger('change');
    }

    /**
     * Format currency values
     */
    function formatCurrency(value) {
        return 'MWK ' + parseFloat(value).toLocaleString('en-US', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    /**
     * Show error messages
     */
    function showError(message) {
        alert(message); // You can replace this with a better error display (toast, modal, etc.)
    }
});
