// Loan Application Form Script

let currentStep = 1;
const totalSteps = 6;
let loanData = {};

$(document).ready(function() {
  initializeForm();
  loadLoanTypes();
  setupEventListeners();
  setupFileUploads();
});

// Initialize form
function initializeForm() {
  updateProgressBar();
  updateNavigationButtons();
  // Set today's date as applied date
  const today = new Date().toISOString().split('T')[0];
  $('#appliedDate').val(today);
}

// Setup event listeners
function setupEventListeners() {
  $('#nextBtn').on('click', nextStep);
  $('#prevBtn').on('click', previousStep);
  $('#loanApplicationForm').on('submit', submitApplication);
  $('#loanType').on('change', calculateLoanSummary);
  $('#loanAmountApplied').on('input', calculateLoanSummary);
  $('#repaymentPeriod').on('input', calculateLoanSummary);
}

// Setup file uploads
function setupFileUploads() {
  // Handle file selection
  $('.file-input').on('change', function() {
    const fileInput = this;
    const wrapper = $(this).closest('.file-upload-wrapper');
    const hint = wrapper.find('.file-upload-hint');
    
    if (fileInput.files && fileInput.files[0]) {
      const fileName = fileInput.files[0].name;
      const fileSize = (fileInput.files[0].size / 1024 / 1024).toFixed(2);
      
      // Show file selected message
      hint.html(`
        <i class="fas fa-file-check" style="color: #27ae60;"></i>
        <p style="color: #27ae60; font-weight: 600;">File selected: ${fileName}</p>
        <small style="color: #666;">${fileSize} MB</small>
      `);
      
      wrapper.css({
        'border-color': '#27ae60',
        'background-color': 'rgba(39, 174, 96, 0.05)'
      });
    }
  });
}

// Load loan types from database
function loadLoanTypes() {
  $.ajax({
    url: 'admin/config/get_loan_types.php',
    type: 'GET',
    dataType: 'json',
    success: function(response) {
      if (response.success && response.data) {
        populateLoanTypes(response.data);
      } else {
        console.error('Failed to load loan types');
      }
    },
    error: function(xhr, status, error) {
      console.error('Error loading loan types:', error);
    }
  });
}

// Populate loan types dropdown
function populateLoanTypes(loanTypes) {
  const select = $('#loanType');
  select.html('<option value="">-- Select Loan Type --</option>');
  
  $.each(loanTypes, function(index, loan) {
    select.append(
      `<option value="${loan.id}" data-interest="${loan.interest_rate_monthly}" data-processing="${loan.processing_fee}" data-insurance="${loan.insurance_fee_yearly}">
        ${loan.loan_name}
      </option>`
    );
  });
}

// Next step
function nextStep() {
  if (validateStep(currentStep)) {
    if (currentStep < totalSteps) {
      currentStep++;
      showStep(currentStep);
    }
  }
}

// Previous step
function previousStep() {
  if (currentStep > 1) {
    currentStep--;
    showStep(currentStep);
  }
}

// Show specific step
function showStep(stepNumber) {
  // Hide all steps
  $('.form-step').removeClass('active');
  
  // Show current step
  $(`.form-step[data-step="${stepNumber}"]`).addClass('active');
  
  // Update progress bar
  updateProgressBar();
  updateNavigationButtons();
  
  // Scroll to form
  $('html, body').animate({
    scrollTop: $('.application-form').offset().top - 100
  }, 300);
}

// Validate step
function validateStep(stepNumber) {
  const stepElement = $(`.form-step[data-step="${stepNumber}"]`);
  const inputs = stepElement.find('input[required], select[required], textarea[required]');
  let isValid = true;

  inputs.each(function() {
    if (!validateField($(this))) {
      isValid = false;
    }
  });

  return isValid;
}

// Validate individual field
function validateField(field) {
  const value = field.val().trim();
  let isValid = true;

  field.removeClass('error');

  if (field.prop('required')) {
    if (field.attr('type') === 'email') {
      isValid = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
    } else if (field.attr('type') === 'tel') {
      isValid = /^[\d\s\-\+\(\)]{7,}$/.test(value);
    } else if (field.attr('type') === 'date') {
      isValid = value !== '';
    } else if (field.attr('type') === 'number') {
      isValid = value !== '' && !isNaN(value);
    } else if (field.attr('type') === 'file') {
      isValid = field[0].files.length > 0;
    } else {
      isValid = value !== '';
    }
  }

  if (!isValid) {
    field.addClass('error');
    showErrorMessage(field, 'This field is required or invalid');
  }

  return isValid;
}

// Show error message
function showErrorMessage(field, message) {
  // Remove existing error message
  field.siblings('.error-message').remove();
  
  // Add new error message
  field.after(`<div class="error-message"><i class="fas fa-exclamation-circle"></i>${message}</div>`);
}

// Update progress bar
function updateProgressBar() {
  const progress = (currentStep / totalSteps) * 100;
  $('#progressBar').css('width', progress + '%');
  
  // Update step indicators
  $('.progress-step').removeClass('active completed');
  
  for (let i = 1; i < currentStep; i++) {
    $(`.progress-step[data-step="${i}"]`).addClass('completed');
  }
  
  $(`.progress-step[data-step="${currentStep}"]`).addClass('active');
}

// Update navigation buttons
function updateNavigationButtons() {
  const prevBtn = $('#prevBtn');
  const nextBtn = $('#nextBtn');
  const submitBtn = $('#submitBtn');

  // Show/hide previous button
  if (currentStep === 1) {
    prevBtn.hide();
  } else {
    prevBtn.show();
  }

  // Show/hide next button and submit button
  if (currentStep === totalSteps) {
    nextBtn.hide();
    submitBtn.show();
  } else {
    nextBtn.show();
    submitBtn.hide();
  }
}

// Calculate loan summary
function calculateLoanSummary() {
  const loanTypeSelect = $('#loanType');
  const loanAmount = parseFloat($('#loanAmountApplied').val()) || 0;
  const repaymentPeriod = parseInt($('#repaymentPeriod').val()) || 12;

  if (!loanTypeSelect.val() || loanAmount < 1000) {
    $('#totalInterest').val('');
    $('#monthlyRepayment').val('');
    $('#totalRepayment').val('');
    return;
  }

  const selectedOption = loanTypeSelect.find(':selected');
  const monthlyInterestRate = parseFloat(selectedOption.data('interest')) / 100;
  const processingFeeRate = parseFloat(selectedOption.data('processing')) / 100;
  const annualInsuranceRate = parseFloat(selectedOption.data('insurance')) / 100;

  // Calculate processing fee (deducted from loan amount)
  const processingFee = loanAmount * processingFeeRate;
  const actualDisbursement = loanAmount - processingFee;

  // Calculate interest (on actual disbursement)
  const totalInterest = actualDisbursement * monthlyInterestRate * repaymentPeriod;

  // Calculate insurance (on requested amount)
  const monthlyInsurance = (loanAmount * annualInsuranceRate) / 100 / 12;
  const totalInsurance = monthlyInsurance * repaymentPeriod;

  // Calculate monthly and total repayment
  const principal = actualDisbursement / repaymentPeriod;
  const monthlyPayment = principal + (totalInterest / repaymentPeriod) + monthlyInsurance;
  const totalRepayment = monthlyPayment * repaymentPeriod;

  // Update display
  $('#totalInterest').val(parseFloat(totalInterest).toFixed(2));
  $('#monthlyRepayment').val(parseFloat(monthlyPayment).toFixed(2));
  $('#totalRepayment').val(parseFloat(totalRepayment).toFixed(2));
}

// Format currency
function formatCurrency(value) {
  return 'MWK ' + parseFloat(value).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

// Submit application
function submitApplication(e) {
  e.preventDefault();

  // Validate final step
  if (!validateStep(currentStep)) {
    alert('Please fill in all required fields');
    return;
  }

  // Prepare form data
  const formData = new FormData($('#loanApplicationForm')[0]);

  // Show loading state
  const submitBtn = $('#submitBtn');
  const originalText = submitBtn.html();
  submitBtn.html('<span class="spinner"></span> Submitting...').prop('disabled', true);

  // Submit via AJAX
  $.ajax({
    url: 'admin/config/save_loan_application.php',
    type: 'POST',
    data: formData,
    processData: false,
    contentType: false,
    dataType: 'json',
    success: function(response) {
      submitBtn.html(originalText).prop('disabled', false);

      if (response.success) {
        // Show success message
        showSuccessMessage(response.application_number);
        
        // Reset form
        $('#loanApplicationForm')[0].reset();
        currentStep = 1;
        showStep(1);
      } else {
        alert('Error: ' + (response.message || 'Failed to submit application'));
      }
    },
    error: function(xhr, status, error) {
      submitBtn.html(originalText).prop('disabled', false);
      console.error('Error:', error);
      alert('An error occurred while submitting your application. Please try again.');
    }
  });
}

// Show success message
function showSuccessMessage(applicationNumber) {
  // Hide form
  $('#loanApplicationForm').fadeOut();
  
  // Show success message
  const successMessage = $(`
    <div class="success-message show" style="display: block;">
      <div style="text-align: center;">
        <i style="font-size: 48px; color: #28a745; margin-bottom: 15px; display: block;">✓</i>
        <h3 style="color: #155724; margin-bottom: 10px;">Application Submitted Successfully!</h3>
        <p style="margin-bottom: 15px;">Your application has been received and is being processed.</p>
        <p style="font-weight: 600; margin-bottom: 20px;">
          <strong>Application Number:</strong> <span style="color: #007b40;">${applicationNumber}</span>
        </p>
        <p style="font-size: 13px; color: #666; margin-bottom: 20px;">
          Please save this number for your records. You can use it to track the status of your application.
        </p>
        <a href="index.php" class="btn btn-primary" style="display: inline-block; margin-top: 10px;">
          <i class="fas fa-home"></i> Back to Home
        </a>
      </div>
    </div>
  `);
  
  $('.application-form').after(successMessage);
}