<?php
session_start();
include_once('../../config/config.php');

// Determine if this is an AJAX request
$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

if ($isAjax) {
    header('Content-Type: application/json');
}

if (!isset($_POST['action'])) {
    if ($isAjax) {
        echo json_encode(['success' => false, 'message' => 'No action specified']);
    } else {
        $_SESSION['error'] = 'No action specified';
        header('Location: savings-products.php');
    }
    exit;
}

$action = $_POST['action'];

try {
    switch ($action) {
        case 'create':
            createProduct($isAjax);
            break;
        case 'update':
            updateProduct($isAjax);
            break;
        case 'delete_product':
            deleteProduct($isAjax);
            break;
        default:
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
            } else {
                $_SESSION['error'] = 'Invalid action';
                header('Location: savings-products.php');
            }
    }
} catch (Exception $e) {
    if ($isAjax) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    } else {
        $_SESSION['error'] = $e->getMessage();
        header('Location: savings-products.php');
    }
}

function createProduct($isAjax = false) {
    global $db;
    
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    
    // Validation
    if (empty($title)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Product title is required']);
        } else {
            $_SESSION['error'] = 'Product title is required';
            header('Location: create-savings-product.php');
        }
        return;
    }
    
    if (empty($description)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Product description is required']);
        } else {
            $_SESSION['error'] = 'Product description is required';
            header('Location: create-savings-product.php');
        }
        return;
    }
    
    try {
        $stmt = $db->prepare("INSERT INTO savings_products (title, description, date) VALUES (:title, :description, NOW())");
        
        $result = $stmt->execute([
            ':title' => $title,
            ':description' => $description
        ]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Savings product created successfully']);
            } else {
                $_SESSION['success'] = 'Savings product created successfully';
                header('Location: create-savings-product.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to create savings product']);
            } else {
                $_SESSION['error'] = 'Failed to create savings product';
                header('Location: create-savings-product.php');
            }
        }
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'UNIQUE constraint failed') !== false || strpos($e->getMessage(), 'Duplicate entry') !== false) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Product title already exists']);
            } else {
                $_SESSION['error'] = 'Product title already exists';
                header('Location: create-savings-product.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
            } else {
                $_SESSION['error'] = 'Database error: ' . $e->getMessage();
                header('Location: create-savings-product.php');
            }
        }
    }
}

function updateProduct($isAjax = false) {
    global $db;
    
    $product_id = intval($_POST['product_id'] ?? 0);
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    
    // Validation
    if ($product_id <= 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
        } else {
            $_SESSION['error'] = 'Invalid product ID';
            header('Location: savings-products.php');
        }
        return;
    }
    
    if (empty($title)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Product title is required']);
        } else {
            $_SESSION['error'] = 'Product title is required';
            header('Location: edit-savings-product.php?id=' . $product_id);
        }
        return;
    }
    
    if (empty($description)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Product description is required']);
        } else {
            $_SESSION['error'] = 'Product description is required';
            header('Location: edit-savings-product.php?id=' . $product_id);
        }
        return;
    }
    
    try {
        // Check if product exists
        $checkStmt = $db->prepare("SELECT product_id FROM savings_products WHERE product_id = :id");
        $checkStmt->execute([':id' => $product_id]);
        
        if ($checkStmt->rowCount() == 0) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Savings product not found']);
            } else {
                $_SESSION['error'] = 'Savings product not found';
                header('Location: savings-products.php');
            }
            return;
        }
        
        // Update product
        $stmt = $db->prepare("UPDATE savings_products SET title = :title, description = :description WHERE product_id = :id");
        
        $result = $stmt->execute([
            ':id' => $product_id,
            ':title' => $title,
            ':description' => $description
        ]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Savings product updated successfully']);
            } else {
                $_SESSION['success'] = 'Savings product updated successfully';
                header('Location: edit-savings-product.php?id=' . $product_id);
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to update savings product']);
            } else {
                $_SESSION['error'] = 'Failed to update savings product';
                header('Location: edit-savings-product.php?id=' . $product_id);
            }
        }
    } catch (PDOException $e) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        } else {
            $_SESSION['error'] = 'Database error: ' . $e->getMessage();
            header('Location: edit-savings-product.php?id=' . $product_id);
        }
    }
}

function deleteProduct($isAjax = false) {
    global $db;
    
    $product_id = intval($_POST['product_id'] ?? 0);
    
    if ($product_id <= 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
        } else {
            $_SESSION['error'] = 'Invalid product ID';
            header('Location: savings-products.php');
        }
        return;
    }
    
    try {
        // Check if product exists
        $checkStmt = $db->prepare("SELECT product_id FROM savings_products WHERE product_id = :id");
        $checkStmt->execute([':id' => $product_id]);
        
        if ($checkStmt->rowCount() == 0) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Savings product not found']);
            } else {
                $_SESSION['error'] = 'Savings product not found';
                header('Location: savings-products.php');
            }
            return;
        }
        
        // Delete product
        $stmt = $db->prepare("DELETE FROM savings_products WHERE product_id = :id");
        $result = $stmt->execute([':id' => $product_id]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Savings product deleted successfully']);
            } else {
                $_SESSION['success'] = 'Savings product deleted successfully';
                header('Location: savings-products.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to delete savings product']);
            } else {
                $_SESSION['error'] = 'Failed to delete savings product';
                header('Location: savings-products.php');
            }
        }
    } catch (PDOException $e) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        } else {
            $_SESSION['error'] = 'Database error: ' . $e->getMessage();
            header('Location: savings-products.php');
        }
    }
}
?>
