<?php
session_start();
include_once('../../config/config.php');

// Determine if this is an AJAX request
$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

if ($isAjax) {
    header('Content-Type: application/json');
}

if (!isset($_POST['action'])) {
    if ($isAjax) {
        echo json_encode(['success' => false, 'message' => 'No action specified']);
    } else {
        $_SESSION['error'] = 'No action specified';
        header('Location: loan_types.php');
    }
    exit;
}

$action = $_POST['action'];

try {
    switch ($action) {
        case 'create':
            createLoanType($isAjax);
            break;
        case 'update':
            updateLoanType($isAjax);
            break;
        case 'delete':
            deleteLoanType($isAjax);
            break;
        case 'get':
            getLoanType($isAjax);
            break;
        default:
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
            } else {
                $_SESSION['error'] = 'Invalid action';
                header('Location: loan_types.php');
            }
    }
} catch (Exception $e) {
    if ($isAjax) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    } else {
        $_SESSION['error'] = $e->getMessage();
        header('Location: loan_types.php');
    }
}

function createLoanType($isAjax = false) {
    global $db;
    
    $loan_name = trim($_POST['loan_name'] ?? '');
    $interest_rate_monthly = floatval($_POST['interest_rate_monthly'] ?? 0);
    $processing_fee = floatval($_POST['processing_fee'] ?? 0);
    $insurance_fee_yearly = floatval($_POST['insurance_fee_yearly'] ?? 0);
    $loan_description = trim($_POST['loan_description'] ?? '');
    
    // Validation
    if (empty($loan_name)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Loan type name is required']);
        } else {
            $_SESSION['error'] = 'Loan type name is required';
            header('Location: loan_types.php');
        }
        return;
    }
    
    if ($interest_rate_monthly < 0 || $processing_fee < 0 || $insurance_fee_yearly < 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Fees and rates must be positive numbers']);
        } else {
            $_SESSION['error'] = 'Fees and rates must be positive numbers';
            header('Location: loan_types.php');
        }
        return;
    }
    
    try {
        $stmt = $db->prepare("INSERT INTO loan_types (loan_name, interest_rate_monthly, processing_fee, insurance_fee_yearly, loan_description) VALUES (:loan_name, :interest_rate_monthly, :processing_fee, :insurance_fee_yearly, :loan_description)");
        
        $result = $stmt->execute([
            ':loan_name' => $loan_name,
            ':interest_rate_monthly' => $interest_rate_monthly,
            ':processing_fee' => $processing_fee,
            ':insurance_fee_yearly' => $insurance_fee_yearly,
            ':loan_description' => $loan_description
        ]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Loan type created successfully']);
            } else {
                $_SESSION['success'] = 'Loan type created successfully';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to create loan type']);
            } else {
                $_SESSION['error'] = 'Failed to create loan type';
                header('Location: loan_types.php');
            }
        }
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'UNIQUE constraint failed') !== false || strpos($e->getMessage(), 'Duplicate entry') !== false) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Loan type name already exists']);
            } else {
                $_SESSION['error'] = 'Loan type name already exists';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
            } else {
                $_SESSION['error'] = 'Database error: ' . $e->getMessage();
                header('Location: loan_types.php');
            }
        }
    }
}

function updateLoanType($isAjax = false) {
    global $db;
    
    $id = intval($_POST['id'] ?? 0);
    $loan_name = trim($_POST['loan_name'] ?? '');
    $interest_rate_monthly = floatval($_POST['interest_rate_monthly'] ?? 0);
    $processing_fee = floatval($_POST['processing_fee'] ?? 0);
    $insurance_fee_yearly = floatval($_POST['insurance_fee_yearly'] ?? 0);
    $loan_description = trim($_POST['loan_description'] ?? '');
    
    // Validation
    if ($id <= 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Invalid loan type ID']);
        } else {
            $_SESSION['error'] = 'Invalid loan type ID';
            header('Location: loan_types.php');
        }
        return;
    }
    
    if (empty($loan_name)) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Loan type name is required']);
        } else {
            $_SESSION['error'] = 'Loan type name is required';
            header('Location: loan_types.php');
        }
        return;
    }
    
    if ($interest_rate_monthly < 0 || $processing_fee < 0 || $insurance_fee_yearly < 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Fees and rates must be positive numbers']);
        } else {
            $_SESSION['error'] = 'Fees and rates must be positive numbers';
            header('Location: loan_types.php');
        }
        return;
    }
    
    try {
        // Check if loan type exists
        $checkStmt = $db->prepare("SELECT id FROM loan_types WHERE id = :id");
        $checkStmt->execute([':id' => $id]);
        
        if ($checkStmt->rowCount() == 0) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Loan type not found']);
            } else {
                $_SESSION['error'] = 'Loan type not found';
                header('Location: loan_types.php');
            }
            return;
        }
        
        // Update loan type
        $stmt = $db->prepare("UPDATE loan_types SET loan_name = :loan_name, interest_rate_monthly = :interest_rate_monthly, processing_fee = :processing_fee, insurance_fee_yearly = :insurance_fee_yearly, loan_description = :loan_description, updated_at = NOW() WHERE id = :id");
        
        $result = $stmt->execute([
            ':id' => $id,
            ':loan_name' => $loan_name,
            ':interest_rate_monthly' => $interest_rate_monthly,
            ':processing_fee' => $processing_fee,
            ':insurance_fee_yearly' => $insurance_fee_yearly,
            ':loan_description' => $loan_description
        ]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Loan type updated successfully']);
            } else {
                $_SESSION['success'] = 'Loan type updated successfully';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to update loan type']);
            } else {
                $_SESSION['error'] = 'Failed to update loan type';
                header('Location: loan_types.php');
            }
        }
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'UNIQUE constraint failed') !== false || strpos($e->getMessage(), 'Duplicate entry') !== false) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Loan type name already exists']);
            } else {
                $_SESSION['error'] = 'Loan type name already exists';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
            } else {
                $_SESSION['error'] = 'Database error: ' . $e->getMessage();
                header('Location: loan_types.php');
            }
        }
    }
}

function deleteLoanType($isAjax = false) {
    global $db;
    
    $id = intval($_POST['id'] ?? 0);
    
    if ($id <= 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Invalid loan type ID']);
        } else {
            $_SESSION['error'] = 'Invalid loan type ID';
            header('Location: loan_types.php');
        }
        return;
    }
    
    try {
        // Check if loan type exists
        $checkStmt = $db->prepare("SELECT id FROM loan_types WHERE id = :id");
        $checkStmt->execute([':id' => $id]);
        
        if ($checkStmt->rowCount() == 0) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Loan type not found']);
            } else {
                $_SESSION['error'] = 'Loan type not found';
                header('Location: loan_types.php');
            }
            return;
        }
        
        // Check if this loan type is used in applications
        $usageStmt = $db->prepare("SELECT COUNT(*) as count FROM individual_loan_applications WHERE loan_type_id = :id");
        $usageStmt->execute([':id' => $id]);
        $usage = $usageStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($usage['count'] > 0) {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Cannot delete this loan type as it is used in ' . $usage['count'] . ' application(s)']);
            } else {
                $_SESSION['error'] = 'Cannot delete this loan type as it is used in ' . $usage['count'] . ' application(s)';
                header('Location: loan_types.php');
            }
            return;
        }
        
        // Delete loan type
        $stmt = $db->prepare("DELETE FROM loan_types WHERE id = :id");
        $result = $stmt->execute([':id' => $id]);
        
        if ($result) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'message' => 'Loan type deleted successfully']);
            } else {
                $_SESSION['success'] = 'Loan type deleted successfully';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Failed to delete loan type']);
            } else {
                $_SESSION['error'] = 'Failed to delete loan type';
                header('Location: loan_types.php');
            }
        }
    } catch (PDOException $e) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        } else {
            $_SESSION['error'] = 'Database error: ' . $e->getMessage();
            header('Location: loan_types.php');
        }
    }
}

function getLoanType($isAjax = false) {
    global $db;
    
    $id = intval($_POST['id'] ?? 0);
    
    if ($id <= 0) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Invalid loan type ID']);
        } else {
            $_SESSION['error'] = 'Invalid loan type ID';
            header('Location: loan_types.php');
        }
        return;
    }
    
    try {
        $stmt = $db->prepare("SELECT * FROM loan_types WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($data) {
            if ($isAjax) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                $_SESSION['error'] = 'Invalid request';
                header('Location: loan_types.php');
            }
        } else {
            if ($isAjax) {
                echo json_encode(['success' => false, 'message' => 'Loan type not found']);
            } else {
                $_SESSION['error'] = 'Loan type not found';
                header('Location: loan_types.php');
            }
        }
    } catch (PDOException $e) {
        if ($isAjax) {
            echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        } else {
            $_SESSION['error'] = 'Database error: ' . $e->getMessage();
            header('Location: loan_types.php');
        }
    }
}
?>
