<?php
session_start();
include_once('../../config/config.php');

if (!isset($_GET['id'])) {
    die("Invalid application ID");
}

$application_id = intval($_GET['id']);

// Fetch application details
$statement = "SELECT * FROM membership_applications WHERE id = :id";

try {
    $stmt = $db->prepare($statement);
    $stmt->execute([':id' => $application_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        die("Application not found");
    }

    // Try to use TCPDF if available, otherwise output HTML for browser print-to-PDF
    $tcpdfPath = __DIR__ . '/../../tcpdf/tcpdf.php';
    
    if (file_exists($tcpdfPath)) {
        // TCPDF is available - use it
        include_once('../../config/tcpdf_config_potrait.php');
        require_once($tcpdfPath);
        
        // Create PDF object
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_PAGE_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        
        // Set document information
        $pdf->SetCreator('Sunbird SACCO');
        $pdf->SetAuthor('Sunbird SACCO');
        $pdf->SetTitle('Membership Application - ' . $row['application_number']);
        $pdf->SetSubject('Membership Application Form');
        
        // Set margins
        $pdf->SetMargins(10, 10, 10);
        $pdf->SetAutoPageBreak(TRUE, 10);
        
        // Add page
        $pdf->AddPage();
        
        // Set font
        $pdf->SetFont('helvetica', '', 10);
        
        // Add header
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->Cell(0, 10, 'SUNBIRD SACCO', 0, 1, 'C');
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->Cell(0, 8, 'MEMBERSHIP APPLICATION FORM', 0, 1, 'C');
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 5, 'Application Number: ' . $row['application_number'], 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 1: Personal Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'PERSONAL INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Full Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['full_name'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Account Type:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst(str_replace('_', ' ', $row['account_type'])), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Branch:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['branch'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'ID Type:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst(str_replace('_', ' ', $row['id_type'])), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'ID Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['id_number'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Date of Birth:', 0, 0, 'L');
        $pdf->Cell(0, 6, !empty($row['date_of_birth']) ? date('M d, Y', strtotime($row['date_of_birth'])) : 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Gender:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst($row['gender']), 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 2: Contact Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'CONTACT INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Contact Address:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['contact_address'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Permanent Address:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['permanent_address'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Village:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['village'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Town:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['town'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'District:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['district'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Phone Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['phone_number'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Email:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['email'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 3: Marital Status
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'MARITAL STATUS', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Marital Status:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst(str_replace('_', ' ', $row['marital_status'])), 0, 1, 'L');
        
        if ($row['marital_status'] === 'married') {
            $pdf->Cell(50, 6, 'Spouse Name:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['spouse_name'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'Spouse ID:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['spouse_id_number'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'Spouse Phone:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['spouse_phone'] ?? 'N/A', 0, 1, 'L');
        }
        
        $pdf->Ln(3);
        
        // Section 4: Occupation Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'OCCUPATION INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Occupation Type:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst(str_replace('_', ' ', $row['occupation_type'])), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Institution/Business:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['institution_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Job Title:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['profession_job_title'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Institution Phone:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['institution_phone'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 5: Monthly Contributions
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'MONTHLY CONTRIBUTIONS (KES)', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Savings:', 0, 0, 'L');
        $pdf->Cell(0, 6, number_format($row['monthly_savings'] ?? 0, 2), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Shares:', 0, 0, 'L');
        $pdf->Cell(0, 6, number_format($row['monthly_shares'] ?? 0, 2), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Fixed Deposit:', 0, 0, 'L');
        $pdf->Cell(0, 6, number_format($row['monthly_fixed_deposit'] ?? 0, 2), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Funeral Cover:', 0, 0, 'L');
        $pdf->Cell(0, 6, number_format($row['kangachepe_funeral_cover'] ?? 0, 2), 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 6: Next of Kin
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'NEXT OF KIN', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Relationship:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin_relationship'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Phone Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin_phone'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Email:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin_email'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 7: Nominated Beneficiaries
        if (!empty($row['nominated_kins'])) {
            $nominated_kins = json_decode($row['nominated_kins'], true);
            if (!empty($nominated_kins)) {
                $pdf->SetFont('helvetica', 'B', 11);
                $pdf->SetFillColor(240, 240, 240);
                $pdf->Cell(0, 7, 'NOMINATED BENEFICIARIES', 0, 1, 'L', true);
                
                $pdf->SetFont('helvetica', '', 9);
                $pdf->SetFillColor(220, 220, 220);
                $pdf->Cell(40, 6, 'Name', 1, 0, 'L', true);
                $pdf->Cell(35, 6, 'Relationship', 1, 0, 'L', true);
                $pdf->Cell(35, 6, 'ID Number', 1, 0, 'L', true);
                $pdf->Cell(25, 6, 'Percentage', 1, 1, 'L', true);
                
                $pdf->SetFont('helvetica', '', 9);
                $pdf->SetFillColor(255, 255, 255);
                foreach ($nominated_kins as $kin) {
                    $pdf->Cell(40, 6, substr($kin['name'] ?? '', 0, 20), 1, 0, 'L');
                    $pdf->Cell(35, 6, substr($kin['relationship'] ?? '', 0, 15), 1, 0, 'L');
                    $pdf->Cell(35, 6, $kin['id'] ?? '', 1, 0, 'L');
                    $pdf->Cell(25, 6, number_format($kin['percentage'] ?? 0, 2) . '%', 1, 1, 'L');
                }
                $pdf->Ln(3);
            }
        }
        
        // Section 8: Who Convinced (Referral Sources)
        $referral_sources = [];
        if (!empty($row['referral_sources'])) {
            $referral_sources = json_decode($row['referral_sources'], true) ?? [];
        }
        
        if (!empty($referral_sources) || !empty($row['referrer_name'])) {
            $pdf->SetFont('helvetica', 'B', 11);
            $pdf->SetFillColor(240, 240, 240);
            $pdf->Cell(0, 7, 'WHO CONVINCED YOU TO JOIN SUNBIRD SACCO?', 0, 1, 'L', true);
            
            $pdf->SetFont('helvetica', '', 10);
            $sources = [];
            if (is_array($referral_sources)) {
                $sources = array_map('ucfirst', $referral_sources);
            }
            $pdf->Cell(50, 6, 'Source(s):', 0, 0, 'L');
            $pdf->Cell(0, 6, implode(', ', $sources) ?: 'N/A', 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'Referrer Name:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['referrer_name'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Ln(3);
        }
        
        // Section 9: Sketch Map
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'SKETCH MAP', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        if (!empty($row['sketch_map'])) {
            $sketchMapPath = __DIR__ . '/../../assets/uploads/' . $row['sketch_map'];
            if (file_exists($sketchMapPath)) {
                $pdf->Cell(0, 6, 'Sketch Map: Attached', 0, 1, 'L');
                $pdf->Ln(2);
                try {
                    $pdf->Image($sketchMapPath, 10, $pdf->GetY(), 190, 100, '', '', 'T', false, 300, 'C');
                    $pdf->Ln(105);
                } catch (Exception $e) {
                    $pdf->Cell(0, 6, 'Error: Could not display sketch map image', 0, 1, 'L');
                }
            } else {
                $pdf->Cell(0, 6, 'Sketch Map: Not found', 0, 1, 'L');
            }
        } else {
            $pdf->Cell(0, 6, 'Sketch Map: No map uploaded', 0, 1, 'L');
        }
        
        $pdf->Ln(3);
        
        // Section 10: Declaration
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'DECLARATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Applicant Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['applicant_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Signature Date:', 0, 0, 'L');
        $pdf->Cell(0, 6, !empty($row['applicant_signature_date']) ? date('M d, Y', strtotime($row['applicant_signature_date'])) : 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Signature:', 0, 0, 'L');
        $pdf->Cell(0, 6, (!empty($row['applicant_signature']) ? 'Attached' : 'N/A'), 0, 1, 'L');
        
        $pdf->Ln(3);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Status:', 0, 0, 'L');
        $pdf->Cell(0, 6, ucfirst($row['application_status']), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Applied Date:', 0, 0, 'L');
        $pdf->Cell(0, 6, date('M d, Y', strtotime($row['created_at'])), 0, 1, 'L');
        
        $pdf->Ln(5);
        
        // Footer
        $pdf->SetFont('helvetica', '', 8);
        $pdf->SetTextColor(128, 128, 128);
        $pdf->Cell(0, 5, 'This is a system-generated document. Generated on: ' . date('M d, Y H:i:s'), 0, 1, 'C');
        
        // Output PDF
        $filename = 'Membership_Application_' . $row['application_number'] . '.pdf';
        $pdf->Output($filename, 'D');
        
    } else {
        // TCPDF not available - output HTML for browser to print to PDF
        $filename = 'Membership_Application_' . $row['application_number'] . '.pdf';
        
        header('Content-Type: text/html; charset=utf-8');
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <title><?php echo $filename; ?></title>
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                body {
                    font-family: Arial, sans-serif;
                    font-size: 11px;
                    line-height: 1.4;
                    color: #333;
                }
                .container {
                    max-width: 8.5in;
                    height: 11in;
                    margin: 0 auto;
                    padding: 0.5in;
                    background: white;
                }
                .header {
                    text-align: center;
                    margin-bottom: 15px;
                    border-bottom: 2px solid #ddd;
                    padding-bottom: 10px;
                }
                .header h1 {
                    font-size: 16px;
                    margin-bottom: 5px;
                }
                .header h2 {
                    font-size: 14px;
                    margin-bottom: 5px;
                }
                .app-number {
                    font-size: 10px;
                    color: #666;
                }
                .section-title {
                    background-color: #e9ecef;
                    padding: 8px;
                    font-weight: bold;
                    margin-top: 15px;
                    margin-bottom: 10px;
                    border-left: 4px solid #007b40;
                }
                .form-row {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 15px;
                    margin-bottom: 10px;
                }
                .form-row.full {
                    grid-template-columns: 1fr;
                }
                .form-row.three {
                    grid-template-columns: 1fr 1fr 1fr;
                }
                .form-group {
                    display: flex;
                    flex-direction: column;
                }
                .form-group label {
                    font-weight: bold;
                    margin-bottom: 3px;
                    font-size: 10px;
                }
                .form-group input,
                .form-group textarea {
                    border: none;
                    border-bottom: 1px solid #999;
                    padding: 4px 0;
                    font-family: Arial, sans-serif;
                    font-size: 10px;
                    background: transparent;
                }
                .form-group textarea {
                    resize: none;
                    min-height: 50px;
                    border: 1px solid #ccc;
                    padding: 5px;
                }
                .footer {
                    margin-top: 20px;
                    text-align: center;
                    font-size: 9px;
                    color: #999;
                    border-top: 1px solid #ddd;
                    padding-top: 10px;
                }
                @media print {
                    body {
                        margin: 0;
                        padding: 0;
                    }
                    .container {
                        margin: 0;
                        padding: 0.5in;
                        height: auto;
                    }
                }
            </style>
        </head>
        <body onload="window.print();">
            <div class="container">
                <div class="header">
                    <h1>SUNBIRD SACCO</h1>
                    <h2>MEMBERSHIP APPLICATION FORM</h2>
                    <p class="app-number">Application Number: <strong><?php echo htmlspecialchars($row['application_number']); ?></strong></p>
                </div>

                <!-- PERSONAL INFORMATION -->
                <div class="section-title">PERSONAL INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Full Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['full_name']); ?>">
                    </div>
                    <div class="form-group">
                        <label>Account Type:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $row['account_type']))); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Branch:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['branch'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>ID Type:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $row['id_type']))); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>ID Number:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['id_number']); ?>">
                    </div>
                    <div class="form-group">
                        <label>Date of Birth:</label>
                        <input type="date" readonly value="<?php echo $row['date_of_birth'] ?? ''; ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Gender:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars(ucfirst($row['gender'])); ?>">
                    </div>
                    <div class="form-group">
                        <label>Marital Status:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $row['marital_status']))); ?>">
                    </div>
                </div>

                <!-- CONTACT INFORMATION -->
                <div class="section-title">CONTACT INFORMATION</div>
                <div class="form-row full">
                    <div class="form-group">
                        <label>Contact Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['contact_address'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row full">
                    <div class="form-group">
                        <label>Permanent Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['permanent_address'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Village:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['village'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Town:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['town'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>District:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['district'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Phone Number:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['phone_number']); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Email Address:</label>
                        <input type="email" readonly value="<?php echo htmlspecialchars($row['email'] ?? ''); ?>">
                    </div>
                </div>

                <!-- OCCUPATION INFORMATION -->
                <div class="section-title">OCCUPATION INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Occupation Type:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $row['occupation_type']))); ?>">
                    </div>
                    <div class="form-group">
                        <label>Institution/Business Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['institution_name'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Job Title/Profession:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['profession_job_title'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Institution Phone:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['institution_phone'] ?? ''); ?>">
                    </div>
                </div>

                <!-- MONTHLY CONTRIBUTIONS -->
                <div class="section-title">MONTHLY CONTRIBUTIONS (MWK)</div>
                <div class="form-row three">
                    <div class="form-group">
                        <label>Savings:</label>
                        <input type="number" readonly value="<?php echo number_format($row['monthly_savings'] ?? 0, 2); ?>">
                    </div>
                    <div class="form-group">
                        <label>Shares:</label>
                        <input type="number" readonly value="<?php echo number_format($row['monthly_shares'] ?? 0, 2); ?>">
                    </div>
                    <div class="form-group">
                        <label>Fixed Deposit:</label>
                        <input type="number" readonly value="<?php echo number_format($row['monthly_fixed_deposit'] ?? 0, 2); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Funeral Cover:</label>
                        <input type="number" readonly value="<?php echo number_format($row['kangachepe_funeral_cover'] ?? 0, 2); ?>">
                    </div>
                </div>

                <!-- NEXT OF KIN -->
                <div class="section-title">NEXT OF KIN</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['next_of_kin_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Relationship:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['next_of_kin_relationship'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Phone Number:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['next_of_kin_phone'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Email Address:</label>
                        <input type="email" readonly value="<?php echo htmlspecialchars($row['next_of_kin_email'] ?? ''); ?>">
                    </div>
                </div>

                <!-- REFERRAL INFORMATION -->
                <?php if (!empty($row['referral_sources']) || !empty($row['referrer_name'])): ?>
                <div class="section-title">REFERRAL INFORMATION</div>
                <div class="form-row full">
                    <table style="width: 100%; border-collapse: collapse; font-size: 11px;">
                        <tbody>
                            <tr>
                                <td style="width: 30%; border: 1px solid #999; padding: 8px; background-color: #f5f5f5;"><strong>How You Heard About Us:</strong></td>
                                <td style="border: 1px solid #999; padding: 8px;">
                                    <?php 
                                        $referral_sources = [];
                                        if (!empty($row['referral_sources'])) {
                                            $referral_sources = json_decode($row['referral_sources'], true) ?? [];
                                        }
                                        if (is_array($referral_sources) && !empty($referral_sources)) {
                                            echo htmlspecialchars(implode(', ', array_map('ucfirst', $referral_sources)));
                                        } else {
                                            echo 'N/A';
                                        }
                                    ?>
                                </td>
                            </tr>
                            <tr>
                                <td style="width: 30%; border: 1px solid #999; padding: 8px; background-color: #f5f5f5;"><strong>Referrer Name:</strong></td>
                                <td style="border: 1px solid #999; padding: 8px;"><?php echo htmlspecialchars($row['referrer_name'] ?? 'N/A'); ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>

                <!-- NOMINATED BENEFICIARIES -->
                <?php if (!empty($row['nominated_kins'])): 
                    $nominated_kins = json_decode($row['nominated_kins'], true);
                    if (!empty($nominated_kins)): ?>
                <div class="section-title">NOMINATED BENEFICIARIES</div>
                <div class="form-row full">
                    <table style="width: 100%; border-collapse: collapse; font-size: 10px;">
                        <thead>
                            <tr style="background-color: #ddd;">
                                <th style="border: 1px solid #999; padding: 5px;">Name</th>
                                <th style="border: 1px solid #999; padding: 5px;">Relationship</th>
                                <th style="border: 1px solid #999; padding: 5px;">ID Number</th>
                                <th style="border: 1px solid #999; padding: 5px;">Percentage</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($nominated_kins as $kin): ?>
                            <tr>
                                <td style="border: 1px solid #999; padding: 5px;"><?php echo htmlspecialchars($kin['name'] ?? ''); ?></td>
                                <td style="border: 1px solid #999; padding: 5px;"><?php echo htmlspecialchars($kin['relationship'] ?? ''); ?></td>
                                <td style="border: 1px solid #999; padding: 5px;"><?php echo htmlspecialchars($kin['id'] ?? ''); ?></td>
                                <td style="border: 1px solid #999; padding: 5px;"><?php echo number_format($kin['percentage'] ?? 0, 2); ?>%</td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; endif; ?>

                <!-- SKETCH MAP -->
                <div class="section-title">SKETCH MAP</div>
                <div class="form-row full">
                    <?php if (!empty($row['sketch_map'])): 
                        $sketchMapPath = __DIR__ . '/../../assets/uploads/' . $row['sketch_map'];
                        if (file_exists($sketchMapPath)): ?>
                    <div style="text-align: center; padding: 10px; border: 1px solid #ccc; border-radius: 4px;">
                        <img src="<?php echo htmlspecialchars($row['sketch_map']); ?>" alt="Sketch Map" style="max-width: 100%; height: auto; max-height: 300px;">
                        <p style="margin-top: 10px; font-size: 9px; color: #666;">Current location to home, business area, or school</p>
                    </div>
                        <?php else: ?>
                    <div style="padding: 10px; background-color: #f0f0f0; border-radius: 4px; text-align: center;">
                        <p style="color: #666;">Sketch map file not found</p>
                    </div>
                        <?php endif; 
                    else: ?>
                    <div style="padding: 10px; background-color: #f0f0f0; border-radius: 4px; text-align: center;">
                        <p style="color: #666;">No sketch map uploaded</p>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- DECLARATION -->
                <div class="section-title">DECLARATION</div>
                <div class="form-row full">
                    <table style="width: 100%; border-collapse: collapse; font-size: 11px;">
                        <tbody>
                            <tr>
                                <td style="width: 30%; border: 1px solid #999; padding: 8px; background-color: #f5f5f5;"><strong>Applicant Name:</strong></td>
                                <td style="border: 1px solid #999; padding: 8px;"><?php echo htmlspecialchars($row['applicant_name'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td style="width: 30%; border: 1px solid #999; padding: 8px; background-color: #f5f5f5;"><strong>Signature Date:</strong></td>
                                <td style="border: 1px solid #999; padding: 8px;"><?php echo !empty($row['applicant_signature_date']) ? date('M d, Y', strtotime($row['applicant_signature_date'])) : 'N/A'; ?></td>
                            </tr>
                            <tr>
                                <td style="width: 30%; border: 1px solid #999; padding: 8px; background-color: #f5f5f5;"><strong>Signature:</strong></td>
                                <td style="border: 1px solid #999; padding: 8px;">
                                    <?php if (!empty($row['applicant_signature'])): ?>
                                        Attached
                                    <?php else: ?>
                                        <span style="color: #999;">Not provided</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <!-- APPLICATION STATUS -->
                <div class="section-title">APPLICATION STATUS</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Status:</label>
                        <strong><?php echo ucfirst($row['application_status']); ?></strong>
                    </div>
                    <div class="form-group">
                        <label>Applied Date:</label>
                        <input type="text" readonly value="<?php echo date('M d, Y', strtotime($row['created_at'])); ?>">
                    </div>
                </div>

                <div class="footer">
                    <p>This is a system-generated document. Generated on: <?php echo date('M d, Y H:i:s'); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
    }

} catch (PDOException $e) {
    die("Error: " . $e->getMessage());
}
?>
