<?php
session_start();
include_once('../../config/config.php');

if (!isset($_GET['id'])) {
    die("Invalid application ID");
}

$application_id = intval($_GET['id']);

// Fetch application details
$statement = "SELECT 
    ila.*,
    lt.loan_name,
    lt.interest_rate_monthly,
    lt.processing_fee,
    lt.insurance_fee_yearly
FROM individual_loan_applications ila
LEFT JOIN loan_types lt ON ila.loan_type_id = lt.id
WHERE ila.id = :id";

try {
    $stmt = $db->prepare($statement);
    $stmt->execute([':id' => $application_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        die("Application not found");
    }

    // Try to use TCPDF if available, otherwise output HTML for browser print-to-PDF
    $tcpdfPath = __DIR__ . '/../../tcpdf/tcpdf.php';
    
    if (file_exists($tcpdfPath)) {
        // TCPDF is available - use it
        include_once('../../config/tcpdf_config_potrait.php');
        require_once($tcpdfPath);
        
        // Create PDF object
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_PAGE_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        
        // Set document information
        $pdf->SetCreator('Sunbird SACCO');
        $pdf->SetAuthor('Sunbird SACCO');
        $pdf->SetTitle('Loan Application - ' . $row['application_number']);
        $pdf->SetSubject('Loan Application Form');
        
        // Set margins
        $pdf->SetMargins(10, 10, 10);
        $pdf->SetAutoPageBreak(TRUE, 10);
        
        // Add page
        $pdf->AddPage();
        
        // Set font
        $pdf->SetFont('helvetica', '', 10);
        
        // Add header
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->Cell(0, 10, 'SUNBIRD SACCO', 0, 1, 'C');
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->Cell(0, 8, 'INDIVIDUAL LOAN APPLICATION FORM', 0, 1, 'C');
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 5, 'Application Number: ' . $row['application_number'], 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 1: Personal Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'PERSONAL INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Full Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['full_name'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'SACCO Account No:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['sacco_account_no'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Email:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['email'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Phone Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['phone_no'], 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Occupation:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['occupation'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Basic Salary:', 0, 0, 'L');
        $pdf->Cell(0, 6, 'KES ' . number_format($row['basic_salary'] ?? 0, 2), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Place of Residence:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['place_of_resident'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 2: Employer Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'EMPLOYER INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Employer Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['employer_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Employment No:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['employment_no'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Employer Address:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['employer_address'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 3: Loan Information
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'LOAN INFORMATION', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Loan Type:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['loan_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Amount Applied:', 0, 0, 'L');
        $pdf->Cell(0, 6, 'KES ' . number_format($row['loan_amount_applied'], 2), 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Repayment Period:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['repayment_period_months'] . ' months', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Monthly Repayment:', 0, 0, 'L');
        $pdf->Cell(0, 6, 'KES ' . number_format($row['monthly_repayment_amount'] ?? 0, 2), 0, 1, 'L');
        
        if ($row['total_repayment_amount']) {
            $pdf->Cell(50, 6, 'Total Repayment:', 0, 0, 'L');
            $pdf->Cell(0, 6, 'KES ' . number_format($row['total_repayment_amount'], 2), 0, 1, 'L');
        }
        
        $pdf->Cell(50, 6, 'Date Applied:', 0, 0, 'L');
        $pdf->Cell(0, 6, date('M d, Y', strtotime($row['applied_date'])), 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 4: Bank Details
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'BANK DETAILS', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Bank Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['bank_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Account Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['account_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Account Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['bank_account_no'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Branch Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['branch_name'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 5: Next of Kin
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'NEXT OF KIN', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(50, 6, 'Name:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Phone Number:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['next_of_kin_phone'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Cell(50, 6, 'Relation:', 0, 0, 'L');
        $pdf->Cell(0, 6, $row['relation'] ?? 'N/A', 0, 1, 'L');
        
        $pdf->Ln(3);
        
        // Section 6: Guarantor Information
        if ($row['guarantor_full_name']) {
            $pdf->SetFont('helvetica', 'B', 11);
            $pdf->SetFillColor(240, 240, 240);
            $pdf->Cell(0, 7, 'GUARANTOR INFORMATION', 0, 1, 'L', true);
            
            $pdf->SetFont('helvetica', '', 10);
            $pdf->Cell(50, 6, 'Full Name:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['guarantor_full_name'], 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'Employer Name:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['guarantor_employer_name'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'Phone Number:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['guarantor_phone_no'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Cell(50, 6, 'ID Number:', 0, 0, 'L');
            $pdf->Cell(0, 6, $row['guarantor_id_no'] ?? 'N/A', 0, 1, 'L');
            
            $pdf->Ln(3);
        }
        
        // Section 7: Assets/Security
        if ($row['assets_description']) {
            $pdf->SetFont('helvetica', 'B', 11);
            $pdf->SetFillColor(240, 240, 240);
            $pdf->Cell(0, 7, 'ASSETS TO BE PLEDGED AS SECURITY', 0, 1, 'L', true);
            
            $pdf->SetFont('helvetica', '', 10);
            $pdf->MultiCell(0, 5, $row['assets_description'], 0, 'L');
            
            $pdf->Ln(3);
        }
        
        // Application Status
        $pdf->SetFont('helvetica', 'B', 11);
        $pdf->SetFillColor(240, 240, 240);
        $pdf->Cell(0, 7, 'APPLICATION STATUS', 0, 1, 'L', true);
        
        $pdf->SetFont('helvetica', '', 10);
        $statusText = ($row['status'] == 'draft') ? 'DRAFT - Not Submitted' : 'SUBMITTED';
        $pdf->Cell(50, 6, 'Status:', 0, 0, 'L');
        $pdf->Cell(0, 6, $statusText, 0, 1, 'L');
        
        $pdf->Ln(5);
        
        // Footer
        $pdf->SetFont('helvetica', '', 8);
        $pdf->SetTextColor(128, 128, 128);
        $pdf->Cell(0, 5, 'This is a system-generated document. Generated on: ' . date('M d, Y H:i:s'), 0, 1, 'C');
        
        // Output PDF
        $filename = 'Loan_Application_' . $row['application_number'] . '.pdf';
        $pdf->Output($filename, 'D');
        
    } else {
        // TCPDF not available - output HTML for browser to print to PDF
        $filename = 'Loan_Application_' . $row['application_number'] . '.pdf';
        
        header('Content-Type: text/html; charset=utf-8');
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <title><?php echo $filename; ?></title>
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                body {
                    font-family: Arial, sans-serif;
                    font-size: 11px;
                    line-height: 1.4;
                    color: #333;
                }
                .container {
                    max-width: 8.5in;
                    height: 11in;
                    margin: 0 auto;
                    padding: 0.5in;
                    background: white;
                }
                .header {
                    text-align: center;
                    margin-bottom: 15px;
                    border-bottom: 2px solid #ddd;
                    padding-bottom: 10px;
                }
                .header h1 {
                    font-size: 16px;
                    margin-bottom: 5px;
                }
                .header h2 {
                    font-size: 14px;
                    margin-bottom: 5px;
                }
                .app-number {
                    font-size: 10px;
                    color: #666;
                }
                .section-title {
                    background-color: #e9ecef;
                    padding: 8px;
                    font-weight: bold;
                    margin-top: 15px;
                    margin-bottom: 10px;
                    border-left: 4px solid #007bff;
                }
                .form-row {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 15px;
                    margin-bottom: 10px;
                }
                .form-row.full {
                    grid-template-columns: 1fr;
                }
                .form-row.three {
                    grid-template-columns: 1fr 1fr 1fr;
                }
                .form-group {
                    display: flex;
                    flex-direction: column;
                }
                .form-group label {
                    font-weight: bold;
                    margin-bottom: 3px;
                    font-size: 10px;
                }
                .form-group input,
                .form-group textarea {
                    border: none;
                    border-bottom: 1px solid #999;
                    padding: 4px 0;
                    font-family: Arial, sans-serif;
                    font-size: 10px;
                    background: transparent;
                }
                .form-group textarea {
                    resize: none;
                    min-height: 50px;
                    border: 1px solid #ccc;
                    padding: 5px;
                }
                .footer {
                    margin-top: 20px;
                    text-align: center;
                    font-size: 9px;
                    color: #999;
                    border-top: 1px solid #ddd;
                    padding-top: 10px;
                }
                @media print {
                    body {
                        margin: 0;
                        padding: 0;
                    }
                    .container {
                        margin: 0;
                        padding: 0.5in;
                        height: auto;
                    }
                }
            </style>
        </head>
        <body onload="window.print();">
            <div class="container">
                <div class="header">
                    <h1>SUNBIRD SACCO</h1>
                    <h2>INDIVIDUAL LOAN APPLICATION FORM</h2>
                    <p class="app-number">Application Number: <strong><?php echo htmlspecialchars($row['application_number']); ?></strong></p>
                </div>

                <!-- PERSONAL INFORMATION -->
                <div class="section-title"><i>👤</i> PERSONAL INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Full Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['full_name']); ?>">
                    </div>
                    <div class="form-group">
                        <label>SACCO Account Number:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['sacco_account_no']); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Email Address:</label>
                        <input type="email" readonly value="<?php echo htmlspecialchars($row['email']); ?>">
                    </div>
                    <div class="form-group">
                        <label>Phone Number:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['phone_no']); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Occupation:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['occupation'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Basic Salary (KES):</label>
                        <input type="number" readonly value="<?php echo number_format($row['basic_salary'] ?? 0, 2); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Place of Residence:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['place_of_resident'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Physical Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['physical_address'] ?? ''); ?>">
                    </div>
                </div>

                <!-- EMPLOYER INFORMATION -->
                <div class="section-title"><i>🏢</i> EMPLOYER INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Employer Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['employer_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Employment Number:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['employment_no'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row full">
                    <div class="form-group">
                        <label>Employer Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['employer_address'] ?? ''); ?>">
                    </div>
                </div>

                <!-- LOAN INFORMATION -->
                <div class="section-title"><i>💰</i> LOAN INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Loan Type:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['loan_name'] ?? 'N/A'); ?>">
                    </div>
                    <div class="form-group">
                        <label>Amount Applied (KES):</label>
                        <input type="number" readonly value="<?php echo number_format($row['loan_amount_applied'], 2); ?>">
                    </div>
                </div>

                <div class="form-row three">
                    <div class="form-group">
                        <label>Repayment Period (Months):</label>
                        <input type="number" readonly value="<?php echo $row['repayment_period_months']; ?>">
                    </div>
                    <div class="form-group">
                        <label>Monthly Repayment (KES):</label>
                        <input type="number" readonly value="<?php echo number_format($row['monthly_repayment_amount'] ?? 0, 2); ?>">
                    </div>
                    <div class="form-group">
                        <label>Total Repayment (KES):</label>
                        <input type="number" readonly value="<?php echo number_format($row['total_repayment_amount'] ?? 0, 2); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Total Interest (KES):</label>
                        <input type="number" readonly value="<?php echo number_format($row['total_interest'] ?? 0, 2); ?>">
                    </div>
                    <div class="form-group">
                        <label>Date Applied:</label>
                        <input type="date" readonly value="<?php echo $row['applied_date']; ?>">
                    </div>
                </div>

                <!-- BANK DETAILS -->
                <div class="section-title"><i>🏦</i> MEMBER BANK DETAILS</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Bank Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['bank_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Account Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['account_name'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Bank Account Number:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['bank_account_no'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Branch Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['branch_name'] ?? ''); ?>">
                    </div>
                </div>

                <!-- NEXT OF KIN -->
                <div class="section-title"><i>👥</i> NEXT OF KIN</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Next of Kin Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['next_of_kin'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Relation:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['relation'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Phone Number:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['next_of_kin_phone'] ?? ''); ?>">
                    </div>
                </div>

                <?php if ($row['guarantor_full_name']): ?>
                <!-- GUARANTOR INFORMATION -->
                <div class="section-title"><i>🛡️</i> GUARANTOR INFORMATION</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Guarantor Full Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_full_name']); ?>">
                    </div>
                    <div class="form-group">
                        <label>Guarantor ID Number:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_id_no'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Employer Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_employer_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Phone Number:</label>
                        <input type="tel" readonly value="<?php echo htmlspecialchars($row['guarantor_phone_no'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label>Postal Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_postal_address'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label>Physical Address:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_physical_address'] ?? ''); ?>">
                    </div>
                </div>
                <?php endif; ?>

                <!-- ASSETS/SECURITY -->
                <div class="section-title"><i>🛡️</i> ASSETS TO BE PLEDGED AS SECURITY</div>
                <div class="form-row full">
                    <div class="form-group">
                        <table style="width: 100%; border-collapse: collapse; font-size: 10px;">
                            <thead>
                                <tr style="background-color: #f5f5f5; border: 1px solid #ccc;">
                                    <th style="width: 8%; border: 1px solid #ccc; padding: 5px; text-align: center;">No</th>
                                    <th style="width: 52%; border: 1px solid #ccc; padding: 5px; text-align: center;">Name of Item; Serial No.; Make; (Other Necessary Description)</th>
                                    <th style="width: 20%; border: 1px solid #ccc; padding: 5px; text-align: center;">Value in MK</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                    $assetRowNum = 1;
                                    $hasAssets = false;
                                    if ($row['assets_description']) {
                                        try {
                                            $assetsData = json_decode($row['assets_description'], true);
                                            if (is_array($assetsData)) {
                                                foreach ($assetsData as $asset) {
                                                    $hasAssets = true;
                                                    $assetName = '';
                                                    $assetValue = '';
                                                    if (is_array($asset)) {
                                                        $assetName = ($asset['name'] ?? $asset['description'] ?? '') .
                                                                    (isset($asset['serial_no']) ? '; S/N: ' . $asset['serial_no'] : '') .
                                                                    (isset($asset['make']) ? '; Make: ' . $asset['make'] : '') .
                                                                    (isset($asset['other_description']) ? '; ' . $asset['other_description'] : '');
                                                        $assetValue = $asset['value'] ?? $asset['amount'] ?? '';
                                                    } else {
                                                        $assetName = $asset;
                                                        $assetValue = '';
                                                    }
                                                    echo '<tr style="border: 1px solid #ccc;"><td style="border: 1px solid #ccc; padding: 5px; text-align: center;">' . $assetRowNum . '</td><td style="border: 1px solid #ccc; padding: 5px;">' . htmlspecialchars($assetName) . '</td><td style="border: 1px solid #ccc; padding: 5px; text-align: right;">' . htmlspecialchars($assetValue) . '</td></tr>';
                                                    $assetRowNum++;
                                                }
                                            } else {
                                                $hasAssets = true;
                                                echo '<tr style="border: 1px solid #ccc;"><td style="border: 1px solid #ccc; padding: 5px; text-align: center;">1</td><td style="border: 1px solid #ccc; padding: 5px;">' . htmlspecialchars($row['assets_description']) . '</td><td style="border: 1px solid #ccc; padding: 5px; text-align: right;"></td></tr>';
                                            }
                                        } catch (Exception $e) {
                                            $hasAssets = true;
                                            echo '<tr style="border: 1px solid #ccc;"><td style="border: 1px solid #ccc; padding: 5px; text-align: center;">1</td><td style="border: 1px solid #ccc; padding: 5px;">' . htmlspecialchars($row['assets_description']) . '</td><td style="border: 1px solid #ccc; padding: 5px; text-align: right;"></td></tr>';
                                        }
                                    }
                                    // If no assets, show 4 empty rows like in the reference image
                                    if (!$hasAssets) {
                                        for ($i = 1; $i <= 4; $i++) {
                                            echo '<tr style="border: 1px solid #ccc; height: 25px;"><td style="border: 1px solid #ccc; padding: 5px; text-align: center;">' . $i . '</td><td style="border: 1px solid #ccc; padding: 5px;"></td><td style="border: 1px solid #ccc; padding: 5px; text-align: right;"></td></tr>';
                                        }
                                    }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- ACCEPTANCE AND DECLARATION -->
                <div class="section-title"><i>✓</i> ACCEPTANCE AND DECLARATION OF APPLICANT</div>
                <p style="font-size: 10px; text-align: justify; margin-bottom: 10px;">
                    I hereby declare that the above particulars are true to the best of my knowledge and belief and that I agree to abide by the conditions of the Society Loaning policy. I further declare that I have understood the terms of the loan. I thereby authorize the necessary deductions to be made from my salary. I fully understand and accept that in case of default, my shares may be used to offset the outstanding loan. Should the monies be insufficient to clear the loans, the item purchased with the loan shall form part of security and Sunbird Tourism SACCO is obliged to take custody thereof including the assets which are pledged as security. All items may be sold to recover expenses.
                </p>

                <div class="form-row three">
                    <div class="form-group">
                        <label>Signature of Applicant:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['full_name']); ?>">
                        <?php if ($row['applicant_signature_date']): ?>
                            <small>Date: <?php echo date('M d, Y', strtotime($row['applicant_signature_date'])); ?></small>
                        <?php endif; ?>
                        <?php if ($row['applicant_signature_file']): ?>
                            <div style="margin-top: 8px; border: 1px solid #ddd; padding: 5px;">
                                
                                <img src="../../../<?php echo htmlspecialchars($row['applicant_signature_file']); ?>" style="max-width: 100%; max-height: 120px; display: block;">
                                <small style="display: block; text-align: center; margin-top: 5px; color: #666;">Signature</small>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="form-group">
                        <label>Witness Name:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['witness_name'] ?? ''); ?>">
                        <?php if ($row['witness_signature_date']): ?>
                            <small>Date: <?php echo date('M d, Y', strtotime($row['witness_signature_date'])); ?></small>
                        <?php endif; ?>
                        <?php if ($row['witness_signature_file']): ?>
                            <div style="margin-top: 8px; border: 1px solid #ddd; padding: 5px;">
                                <img src="../../../<?php echo htmlspecialchars($row['witness_signature_file']); ?>" style="max-width: 100%; max-height: 120px; display: block;">
                                <small style="display: block; text-align: center; margin-top: 5px; color: #666;">Signature</small>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="form-group">
                        <label>HR/Finance Manager:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['hr_finance_manager_name'] ?? ''); ?>">
                        <?php if ($row['hr_finance_manager_date']): ?>
                            <small>Date: <?php echo date('M d, Y', strtotime($row['hr_finance_manager_date'])); ?></small>
                        <?php endif; ?>
                        <?php if ($row['hr_finance_manager_signature_file']): ?>
                            <div style="margin-top: 8px; border: 1px solid #ddd; padding: 5px;">
                                <img src="../../../<?php echo htmlspecialchars($row['hr_finance_manager_signature_file']); ?>" style="max-width: 100%; max-height: 120px; display: block;">
                                <small style="display: block; text-align: center; margin-top: 5px; color: #666;">Signature</small>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <?php if ($row['loan_recommendation']): ?>
                <div class="form-row full">
                    <div class="form-group" style="margin-top: 10px;">
                        <label><strong>Loan Recommendations:</strong></label>
                        <p style="font-size: 10px; padding: 5px; border: 1px solid #ccc; background-color: #f9f9f9;">
                            <?php echo htmlspecialchars($row['loan_recommendation']); ?>
                        </p>
                    </div>
                </div>
                <?php endif; ?>

                <!-- LOAN GUARANTOR -->
                <?php if ($row['guarantor_full_name']): ?>
                <div class="section-title" style="margin-top: 15px;"><i>🛡️</i> LOAN GUARANTOR</div>
                <table style="width: 100%; border-collapse: collapse; font-size: 10px;">
                    <tbody>
                        <tr style="border: 1px solid #ccc;">
                            <td style="width: 50%; border: 1px solid #ccc; padding: 5px;"><strong>Full Name:</strong><br><?php echo htmlspecialchars($row['guarantor_full_name']); ?></td>
                            <td style="width: 50%; border: 1px solid #ccc; padding: 5px;"><strong>Employer Name:</strong><br><?php echo htmlspecialchars($row['guarantor_employer_name'] ?? ''); ?></td>
                        </tr>
                        <tr style="border: 1px solid #ccc;">
                            <td style="border: 1px solid #ccc; padding: 5px;"><strong>Postal Address:</strong><br><?php echo htmlspecialchars($row['guarantor_postal_address'] ?? ''); ?></td>
                            <td style="border: 1px solid #ccc; padding: 5px;"><strong>Physical Address:</strong><br><?php echo htmlspecialchars($row['guarantor_physical_address'] ?? ''); ?></td>
                        </tr>
                        <tr style="border: 1px solid #ccc;">
                            <td style="border: 1px solid #ccc; padding: 5px;"><strong>Phone No:</strong><br><?php echo htmlspecialchars($row['guarantor_phone_no'] ?? ''); ?></td>
                            <td style="border: 1px solid #ccc; padding: 5px;"><strong>ID No:</strong><br><?php echo htmlspecialchars($row['guarantor_id_no'] ?? ''); ?></td>
                        </tr>
                    </tbody>
                </table>

                <p style="font-size: 10px; margin-top: 10px; text-align: justify; background-color: #fff3cd; padding: 8px; border-left: 3px solid #ff9800;">
                    <strong>Guarantor's Oath:</strong> I hereby guarantee that the above information is correct and that Mr./Mrs./Miss shall repay the loan irrevocably undertake to REPAY to SUNBIRD SACCO LIMITED the value of the loan on any form of indebtedness and other associated cost of debt recovery/charges on behalf of the above mentioned applicant if he/she fails to repay the loan on or before the due date.
                </p>

                <div class="form-row" style="margin-top: 10px;">
                    <div class="form-group">
                        <label>Guarantor Signature:</label>
                        <input type="text" readonly value="<?php echo htmlspecialchars($row['guarantor_full_name'] ?? ''); ?>">
                        <?php if ($row['guarantor_signature_date']): ?>
                            <small>Date: <?php echo date('M d, Y', strtotime($row['guarantor_signature_date'])); ?></small>
                        <?php endif; ?>
                        <?php if ($row['guarantor_signature_file']): ?>
                            <div style="margin-top: 8px; border: 1px solid #ddd; padding: 5px;">
                                <img src="../../uploads/<?php echo htmlspecialchars($row['guarantor_signature_file']); ?>" style="max-width: 100%; max-height: 120px; display: block;">
                                <small style="display: block; text-align: center; margin-top: 5px; color: #666;">Signature</small>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <!-- APPLICATION STATUS -->
                <div class="section-title"><i>ℹ️</i> APPLICATION STATUS</div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Status:</label>
                        <strong><?php echo ($row['status'] == 'draft') ? 'DRAFT - Not Submitted' : 'SUBMITTED'; ?></strong>
                    </div>
                    <div class="form-group">
                        <label>Generated Date:</label>
                        <input type="text" readonly value="<?php echo date('M d, Y H:i:s'); ?>">
                    </div>
                </div>

                <div class="footer">
                    <p>This is a system-generated document. Generated on: <?php echo date('M d, Y H:i:s'); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
    }

} catch (PDOException $e) {
    die("Error: " . $e->getMessage());
}
?>
   
